'use strict';

logging.controllers.module('logs-overview', ['xhr', 'loading_spinner'], function () {
	return {
		vue: true,
		name: 'gx-logs-overview',
		template: '#logs-overview',
		data: function data() {
			return {
				logs: [],
				logfiles: [],
				selectedLogCategory: '',
				selectedLogFileIndex: 0,
				maxSelectableLogFiles: 0,
				minLogLevel: 300,
				displayedLogLevels: ['DEBUG', 'INFO', 'NOTICE', 'WARNING', 'ERROR', 'CRITICAL', 'ALERT', 'EMERGENCY'],
				showNoLogsAvailable: true,
				deleteSelectedLogfileModalActive: false,
				lang: {
					labels: {
						noLogsToShow: jse.core.lang.translate('labels_no_logs_to_show', 'logging'),
						category: jse.core.lang.translate('labels_category', 'logging'),
						file: jse.core.lang.translate('labels_file', 'logging'),
						logLevel: jse.core.lang.translate('labels_log_level', 'logging'),
						logLevels: {
							debug: jse.core.lang.translate('labels_log_levels_debug', 'logging'),
							info: jse.core.lang.translate('labels_log_levels_info', 'logging'),
							notice: jse.core.lang.translate('labels_log_levels_notice', 'logging'),
							warning: jse.core.lang.translate('labels_log_levels_warning', 'logging'),
							error: jse.core.lang.translate('labels_log_levels_error', 'logging'),
							critical: jse.core.lang.translate('labels_log_levels_critical', 'logging'),
							alert: jse.core.lang.translate('labels_log_levels_alert', 'logging'),
							emergency: jse.core.lang.translate('labels_log_levels_emergency', 'logging')
						}
					},
					buttons: {
						deleteSelectedCategory: jse.core.lang.translate('buttons_delete_selected_category', 'logging'),
						deleteOldLogs: jse.core.lang.translate('buttons_delete_old_logs', 'logging'),
						deleteAllLogs: jse.core.lang.translate('buttons_delete_all_logs', 'logging'),
						showNextLogFile: jse.core.lang.translate('buttons_show_next_log_file', 'logging')
					},
					modals: {
						deleteSingleLogfile: {
							title: jse.core.lang.translate('modal_title_delete_log_files_by_category', 'logging'),
							text: jse.core.lang.translate('modal_text_delete_log_files_by_category', 'logging'),
							close: jse.core.lang.translate('close', 'buttons'),
							confirm: jse.core.lang.translate('delete', 'buttons')
						},
						deleteOldLogfiles: {
							title: jse.core.lang.translate('modal_title_delete_old_log_files', 'logging'),
							text: jse.core.lang.translate('modal_text_delete_old_log_files', 'logging'),
							days: jse.core.lang.translate('modal_label_delete_old_log_files_days', 'logging'),
							age: jse.core.lang.translate('modal_label_delete_old_log_files_age', 'logging'),
							close: jse.core.lang.translate('close', 'buttons'),
							confirm: jse.core.lang.translate('delete', 'buttons')
						},
						deleteAllLogfiles: {
							title: jse.core.lang.translate('modal_title_delete_all_log_files', 'logging'),
							text: jse.core.lang.translate('modal_text_delete_all_log_files', 'logging'),
							close: jse.core.lang.translate('close', 'buttons'),
							confirm: jse.core.lang.translate('delete', 'buttons')
						}
					}
				},
				days: 14
			};
		},
		methods: {
			loadLogfiles: function loadLogfiles() {
				var vm = this;
				var $target = $('.logs-overview');
				var spinner = jse.libs.loading_spinner.show($target);
				this.logfiles = [];
				this.selectedLogFileIndex = 0;
				this.selectedLogCategory = '';
				this.maxSelectableLogFiles = 0;

				jse.libs.xhr.get({ url: './admin.php?do=Logging/GetLogfiles' }).done(function (response) {
					if (response.logfiles === undefined || response.logfiles.length === 0) {
						vm.logfiles = {};
						vm.selectedLogFileIndex = 0;
						vm.selectedLogCategory = '';
						vm.maxSelectableLogFiles = 0;
						vm.logs = [];
						vm.showNoLogsAvailable = true;
						jse.libs.loading_spinner.hide(spinner);
						return;
					}

					vm.logfiles = response.logfiles;
					vm.selectedLogFileIndex = 0;
					vm.selectedLogCategory = Object.keys(response.logfiles)[0];
					vm.maxSelectableLogFiles = vm.logfiles[vm.selectedLogCategory].length - 1;
					vm.loadLogs();

					jse.libs.loading_spinner.hide(spinner);
				}).fail(function (response) {
					return console.error(response);
				});
			},
			loadLogs: function loadLogs() {
				var vm = this;
				var $target = $('.logs-overview');
				var spinner = jse.libs.loading_spinner.show($target);
				var logFile = this.logfiles[this.selectedLogCategory][this.selectedLogFileIndex];
				this.logs = [];

				jse.libs.xhr.get({ url: './admin.php?do=Logging/GetLogs&logfile=' + logFile }).done(function (response) {
					var _iteratorNormalCompletion = true;
					var _didIteratorError = false;
					var _iteratorError = undefined;

					try {
						for (var _iterator = response.logs[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
							var log = _step.value;

							try {
								vm.logs.push(JSON.parse(log));
							} catch (error) {
								$('.message-stack').append('\n\t\t\t\t\t\t\t\t\t\t<div class="alert alert-danger">\n\t\t\t\t\t\t\t\t\t\t<button type="button" class="close" data-dismiss="alert">\xD7</button>\n\t\t\t\t\t\t\t\t\t\t' + jse.core.lang.translate('labels_json_error_first_part', 'logging') + logFile + jse.core.lang.translate('labels_json_error_last_part', 'logging') + '\n\t\t\t\t\t\t\t\t\t\t</div>\n\t\t\t\t\t\t\t\t\t');
							}
						}
					} catch (err) {
						_didIteratorError = true;
						_iteratorError = err;
					} finally {
						try {
							if (!_iteratorNormalCompletion && _iterator.return) {
								_iterator.return();
							}
						} finally {
							if (_didIteratorError) {
								throw _iteratorError;
							}
						}
					}

					vm.checkForNoLogsNotice();
					jse.libs.loading_spinner.hide(spinner);
				}).fail(function (response) {
					return console.error(response);
				});
			},
			checkForNoLogsNotice: function checkForNoLogsNotice() {
				this.showNoLogsAvailable = true;
				var _iteratorNormalCompletion2 = true;
				var _didIteratorError2 = false;
				var _iteratorError2 = undefined;

				try {
					for (var _iterator2 = this.logs[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
						var log = _step2.value;

						if (this.displayedLogLevels.includes(log.level_name)) {
							this.showNoLogsAvailable = false;
						}
					}
				} catch (err) {
					_didIteratorError2 = true;
					_iteratorError2 = err;
				} finally {
					try {
						if (!_iteratorNormalCompletion2 && _iterator2.return) {
							_iterator2.return();
						}
					} finally {
						if (_didIteratorError2) {
							throw _iteratorError2;
						}
					}
				}
			},
			changedLogCategory: function changedLogCategory() {
				this.maxSelectableLogFiles = this.logfiles[this.selectedLogCategory].length - 1;
				this.selectedLogFileIndex = 0;
				this.loadLogs();
			},
			showNextLogFile: function showNextLogFile() {
				$('html, body').animate({
					scrollTop: 0
				});
				this.selectedLogFileIndex++;
				this.loadLogs();
			},
			showDeleteSelectedLogfileModal: function showDeleteSelectedLogfileModal() {
				$('.delete-single-log.modal').modal('show');
			},
			deleteSelectedLogfile: function deleteSelectedLogfile() {
				var _this = this;

				var vm = this;
				var selected = this.logfiles[this.selectedLogCategory][0];

				jse.libs.xhr.post({
					url: './admin.php?do=Logging/DeleteSelectedLogs',
					data: { selected: selected }
				}).done(function (response) {
					vm.loadLogfiles();
				}).fail(function (response) {
					return console.error(response, 'Could not delete selected logs: ' + _this.selectedLogCategory);
				});

				$('.delete-single-log.modal').modal('hide');
			},
			showDeleteOldLogfilesModal: function showDeleteOldLogfilesModal() {
				$('.delete-old-logs.modal').modal('show');
			},
			deleteOldLogfiles: function deleteOldLogfiles() {
				var vm = this;

				jse.libs.xhr.post({
					url: './admin.php?do=Logging/DeleteOldLogfiles',
					data: { days: this.days }
				}).done(function (response) {
					vm.loadLogfiles();
				}).fail(function (response) {
					return console.error(response, 'Could not delete old logs');
				});

				$('.delete-old-logs.modal').modal('hide');
			},
			showDeleteAllLogfilesModal: function showDeleteAllLogfilesModal() {
				$('.delete-all-logs.modal').modal('show');
			},
			deleteAllLogfiles: function deleteAllLogfiles() {
				var vm = this;

				jse.libs.xhr.post({ url: './admin.php?do=Logging/DeleteAllLogfiles' }).done(function (response) {
					vm.loadLogfiles();
				}).fail(function (response) {
					return console.error(response, 'Could not delete all log files');
				});

				$('.delete-all-logs.modal').modal('hide');
			}
		},
		mounted: function mounted() {
			this.loadLogfiles();
		},

		init: function init(done) {
			return done();
		}
	};
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
